#' Creation du graphique du taux d'evolution 2016/20 (%) de l'artificialisation cadastree par aire d'attraction des villes.
#' @description Graphique du taux d'evolution 2016/20 (%) de l'artificialisation cadastree par aire d'attraction des villes.
#'
#' @param millesime_stock_artif une année parmi les millesimes sélectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param code_reg code insee de la région sur laquelle construire le graphique
#' @return Un diagramme en barres
#'
#' @importFrom attempt stop_if stop_if_not
#' @importFrom COGiter charger_zonage ajouter_typologie filtrer_cog
#' @importFrom dplyr filter mutate select group_by lag ungroup
#' @importFrom forcats fct_drop fct_inorder fct_rev
#' @importFrom ggplot2 ggplot aes geom_bar geom_label position_dodge coord_flip labs theme element_text scale_y_continuous
#' @importFrom glue glue
#' @importFrom lubridate year
#' @importFrom mapfactory format_fr_pct_2
#' @importFrom scales percent_format
#' @importFrom stringr str_detect
#'
#' @export
#' @examples
#' creer_graphe_2_6(millesime_stock_artif = 2020, millesime_population = 2019, code_reg = "52")

creer_graphe_2_6 <- function(millesime_stock_artif = NULL, millesime_population = NULL, code_reg = NULL) {

  attempt::stop_if(millesime_stock_artif, is.null, msg = "millesime_stock_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_stock_artif, is.numeric, msg = "millesime_stock_artif n\'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  millesime_debut <- 2016

  # seuils population
  pop_3 <- 40000
  pop_2 <- 10000
  pop_1 <- 2000

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # creation de la fonction inverse
  `%notin%` <- Negate(`%in%`)

  if (code_reg %in% c('52')) {

  # population du millesime
  population <- population_legale %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$date == millesime_population,
                  .data$TypeZone =="Communes") %>%
    dplyr::mutate(seuil_pop = dplyr::case_when(
      .data$population_municipale > pop_3  ~ glue::glue("communes de plus\nde ",pop_3," habitants"),
      .data$population_municipale > pop_2  ~ glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
      .data$population_municipale > pop_1  ~ glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
      .data$population_municipale > 0  ~ glue::glue("communes de moins\nde ",pop_1," habitants"),
      TRUE ~ "")
      ) %>%
    dplyr::mutate(seuil_code = dplyr::case_when(
      .data$population_municipale > pop_3  ~ "D",
      .data$population_municipale > pop_2  ~ "C",
      .data$population_municipale > pop_1  ~ "B",
      .data$population_municipale > 0  ~ "A",
      TRUE ~ "")
    ) %>%
    dplyr::rename ("population_n"="population_municipale") %>%
    dplyr::select (-.data$date)

  # table des seuils
  seuil_population <- population %>%
    dplyr::select (.data$CodeZone, .data$seuil_pop)

  # population du millesime
  population_n <- population %>%
    dplyr::select (.data$seuil_pop, .data$seuil_code, .data$population_n) %>%
    dplyr::group_by(.data$seuil_pop, .data$seuil_code) %>%
    dplyr::summarise(population_n = sum(.data$population_n, na.rm = T)) %>%
    dplyr::ungroup()


  # preparation des donnees

  data <- COGiter::filtrer_cog(stock_artificialise,
                               reg = code_reg) %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::filter(.data$TypeZone == "Communes") %>%
    dplyr::mutate(date = lubridate::year(.data$date)) %>%
    dplyr::filter(.data$date == millesime_stock_artif | .data$date == millesime_debut) %>%
    dplyr::select (-.data$surf_cadastree) %>%
    dplyr::group_by(.data$seuil_pop,.data$date) %>%
    dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::group_by(.data$seuil_pop) %>%
    dplyr::mutate(evolution = (.data$surface_artificialisee - dplyr::lag(.data$surface_artificialisee))*100/ dplyr::lag(.data$surface_artificialisee)) %>%
    dplyr::ungroup() %>%
    dplyr::filter(.data$date == millesime_stock_artif) %>%
    dplyr::mutate(seuil_pop = factor(.data$seuil_pop, levels = c(glue::glue("communes de moins\nde ",pop_1," habitants"), # çà n'a pas d'effet !
                                                                 glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
                                                                 glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
                                                                 glue::glue("communes de plus\nde ",pop_3," habitants"))) %>%
                    forcats::fct_rev()) %>%
    dplyr::select(.data$seuil_pop,.data$evolution)


  valeur_max <- max(data$evolution, na.rm = T)

  graph_2_6<-data  %>%
    ggplot2::ggplot(ggplot2::aes(x = .data$seuil_pop, y = .data$evolution)) +
    ggplot2::geom_bar(stat = "identity", fill = "#FF8D7E", width = .5) +
    ggplot2::geom_label(ggplot2::aes(y = .data$evolution , label=mapfactory::format_fr_pct_2(.data$evolution,0.1)),
                        position= ggplot2::position_dodge(width=1),
                        vjust=0.5,
                        hjust=-0.5,
                        size=3,
                        color ="#FF8D7E",
                        fill = "#FF8D7E30") +
    ggplot2::coord_flip(expand = FALSE) +
    ggplot2::labs(title=glue::glue("\u00c9volution de la surface cadastr\u00e9e consomm\u00e9e\nentre le 1/1/{millesime_debut} et le 1/1/{millesime_stock_artif} (en %)"),
                  subtitle = "",
                  x = "",
                  y = "",
                  fill = "",
                  caption = glue::glue("Source : Cerema Fichiers fonciers")) +
    ggplot2::theme(axis.text.x = ggplot2::element_text(), legend.position = "none") +
    ggplot2::scale_y_continuous(labels = scales::percent_format(accuracy = 0.1, big.mark = " ", scale=1), limits = c(0,valeur_max*1.2))

  return(graph_2_6)
  }
  if (code_reg  %notin% c('52')) {
    return(invisible(NULL))
  }
  }
