#' Creation du graphique de l artificialisation par destination en ha sur 10 ans
#' @description Graphique de l artificialisation par destination en ha sur 10 ans
#'
#' @param millesime_obs_artif une année parmi les millésimes sélectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#'
#' @return cinq diagrammes en barres
#'
#' @importFrom gouvdown scale_fill_gouv_discrete
#' @importFrom COGiter filtrer_cog
#' @importFrom dplyr filter select mutate arrange
#' @importFrom forcats fct_drop fct_relevel
#' @importFrom ggplot2 ggplot geom_bar aes position_dodge labs facet_wrap theme
#' @importFrom glue glue
#' @importFrom lubridate year
#' @importFrom tidyr gather
#' @importFrom attempt stop_if stop_if_not
#'
#' @export
#'
#' @examples
#' creer_graphe_2_5(millesime_obs_artif = 2019 , code_reg = "52")
creer_graphe_2_5 <- function(millesime_obs_artif=NULL, code_reg = NULL) {

  attempt::stop_if(millesime_obs_artif, is.null, msg = "millesime_obs_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_obs_artif, is.numeric, msg = "millesime_obs_artif n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  # calcul des millesimes extremes du graphique
  millesime_debut <- millesime_obs_artif - 10
  millesime_fin <- millesime_obs_artif - 1

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # Creation de la table utile a la production du graphique
  data <- observatoire_artificialisation %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone == "D\u00e9partements") %>%
    dplyr::arrange(.data$CodeZone) %>%
    dplyr::mutate(Zone=factor(.data$Zone) %>% forcats::fct_inorder())%>%
    dplyr::select(.data$Zone, .data$date, .data$flux_naf_artificialisation_activite, .data$flux_naf_artificialisation_habitation, .data$flux_naf_artificialisation_mixte) %>%
    dplyr::mutate(Zone = forcats::fct_drop(.data$Zone)) %>%
    dplyr::arrange(.data$Zone) %>%
    dplyr::mutate(date = as.character(lubridate::year(.data$date - 1))) %>%
    dplyr::filter(.data$date < millesime_obs_artif, .data$date > millesime_obs_artif - 11) %>%
    # conserve les 10 derniers millesimes
    tidyr::gather(variable, valeur, .data$flux_naf_artificialisation_activite:.data$flux_naf_artificialisation_mixte) %>%
    dplyr::mutate(valeur = .data$valeur / 10000) %>%
    dplyr::mutate(
      variable = replace(.data$variable, .data$variable == "flux_naf_artificialisation_activite", "activit\u00e9"),
      variable = replace(.data$variable, .data$variable == "flux_naf_artificialisation_habitation", "habitation"),
      variable = replace(.data$variable, .data$variable == "flux_naf_artificialisation_mixte", "mixte")
    )

  # creation du graphique

  graph_2_5 <- data %>%
    ggplot2::ggplot() +
    ggplot2::geom_bar(ggplot2::aes(x = .data$date, y = .data$valeur, fill = .data$variable),
      width = 0.6, stat = "identity", position = ggplot2::position_dodge()
    ) +
    ggplot2::labs(
      title = glue::glue("Consommation d\u2019espaces cadastr\u00e9s par destination,  par d\u00e9partement de {millesime_debut} \u00e0 {millesime_fin} (10 ans)", width = 60),
      subtitle = "<span style = \'color:#169B62\'> activit\u00e9</span>,<span style = \'color:#FF8D7E\'> habitation</span> et <span style = \'color:#7D4E5B\'> mixte</span>",
      x = "", y = "",
      fill = "",
      caption = glue::glue("Source : Observatoire de l\'artificialisation {millesime_obs_artif}")
    ) +
    ggplot2::facet_grid(Zone ~ .) +
    gouvdown::scale_fill_gouv_discrete(palette = "pal_gouv_qual2") +
    ggplot2::theme(
      legend.position = "none",
      plot.subtitle = ggtext::element_markdown(size = 12, lineheight = 1.2)
    )

  return(graph_2_5)
}
