#' Creation du graphique de la consommation d espaces NAF sur 10 ans en ha
#' @description Graphiques en barres de la consommation d espaces naturels agricoles et forestiers sur 10 ans en ha par departement
#'
#' @param millesime_obs_artif une année parmi les millesimes sélectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#'
#' @return cinq diagrammes en barres
#'
#' @importFrom dplyr filter select mutate arrange
#' @importFrom COGiter filtrer_cog
#' @importFrom forcats fct_drop fct_relevel fct_inorder
#' @importFrom ggplot2 ggplot geom_bar aes geom_text labs facet_wrap theme element_text scale_y_continuous
#' @importFrom gouvdown scale_fill_gouv_discrete
#' @importFrom glue glue
#' @importFrom lubridate year
#' @importFrom stringr str_wrap
#' @importFrom mapfactory format_fr
#' @importFrom attempt stop_if stop_if_not
#'
#' @export
#'
#' @examples
#' creer_graphe_2_4(millesime_obs_artif = 2020 ,code_reg = '52')
creer_graphe_2_4 <- function(millesime_obs_artif = NULL, code_reg = NULL) {

  attempt::stop_if(millesime_obs_artif, is.null, msg = "millesime_obs_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_obs_artif, is.numeric, msg = "millesime_obs_artif n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  # calcul des millesimes extremes du graphique
  millesime_debut <- millesime_obs_artif - 10
  millesime_fin <- millesime_obs_artif - 1

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # Creation de la table utile a la production du graphique
  data <- observatoire_artificialisation %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone == "D\u00e9partements") %>%
    dplyr::arrange(.data$CodeZone) %>%
    dplyr::mutate(Zone=factor(.data$Zone) %>% forcats::fct_inorder())%>%
    dplyr::select(.data$Zone, .data$date, .data$flux_naf_artificialisation_total) %>%
    dplyr::mutate(
      Zone = forcats::fct_drop(.data$Zone),
      flux_naf_artificialisation_total = .data$flux_naf_artificialisation_total / 10000
    ) %>%
    dplyr::arrange(.data$Zone) %>%
    dplyr::mutate(date = as.character(lubridate::year(.data$date - 1))) %>%
    dplyr::filter(.data$date < millesime_obs_artif, .data$date > millesime_obs_artif - 11) # conserve les 10 derniers millesimes

  # creation du graphique
  graph_2_4 <- data %>%
    ggplot2::ggplot() +
    ggplot2::geom_bar(ggplot2::aes(x = .data$date, y = .data$flux_naf_artificialisation_total, fill = .data$Zone),
      width = 0.6, stat = "identity"
    ) +
    ggplot2::geom_text(ggplot2::aes(x = .data$date, y = .data$flux_naf_artificialisation_total - 120, label = mapfactory::format_fr(.data$flux_naf_artificialisation_total, 0), group = .data$Zone), color = "white", size = 3) +
    ggplot2::labs(
      title = stringr::str_wrap("Consommation d\'espaces naturels, agricoles et forestiers (Naf) par d\u00e9partement", width = 60),
      subtitle = glue::glue("de {millesime_debut} \u00e0 {millesime_fin} (10 ans)"),
      x = "", y = "Surface NAF consomm\u00e9es (en ha)",
      fill = "",
      caption = glue::glue("Source : Observatoire de l\'artificialisation {millesime_obs_artif}")
    ) +
    ggplot2::facet_wrap(Zone ~ ., ncol = 1) +
    ggplot2::theme(legend.position = "none") +
    gouvdown::scale_fill_gouv_discrete(palette = "pal_gouv_qual2") +
    ggplot2::scale_y_continuous(labels = function(x) format(x, big.mark = " "))

  return(graph_2_4)
}
