#' Creation du graphique de l artificialisation OCSGE par type de territoire.
#' @description Graphique de l artificialisation OCSGE par type de territoire.
#'
#' @param millesime_ocsge une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#' @return Un graphique en barres
#' @import COGiter
#'
#' @importFrom COGiter charger_zonage ajouter_typologie filtrer_cog
#' @importFrom dplyr mutate filter select group_by summarise ungroup arrange
#' @importFrom ggplot2 ggplot aes geom_point position_dodge geom_linerange geom_label coord_flip scale_y_continuous labs theme element_blank element_text scale_color_manual
#' @importFrom ggtext element_textbox_simple
#' @importFrom glue glue
#' @importFrom lubridate year as_date
#' @importFrom mapfactory format_fr_pct_2
#' @importFrom scales label_number
#' @importFrom tidyr pivot_longer
#' @importFrom forcats fct_drop fct_inorder fct_rev
#' @importFrom attempt stop_if stop_if_not
#'
#' @export
#'
#' @examples
#' creer_graphe_1_6(millesime_ocsge = 2016, millesime_population = 2019, code_reg = '52')
creer_graphe_1_6 <- function(millesime_ocsge = NULL, millesime_population = NULL, code_reg = NULL) {

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }
  attempt::stop_if(millesime_ocsge, is.null, msg = "millesime_ocsge n'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_ocsge, is.numeric, msg = "millesime_ocsge n'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n'est pas renseign\u00e9")

  # creation de la fonction inverse
  `%notin%` <- Negate(`%in%`)

  # seuils population
  pop_3 <- 40000
  pop_2 <- 10000
  pop_1 <- 2000

  if (code_reg == '52') {

    # population du millesime
    population <- population_legale %>%
      dplyr::mutate(date=lubridate::year(.data$date)) %>%
      COGiter::filtrer_cog(reg = code_reg) %>%
      dplyr::filter(.data$date == millesime_population,
                    .data$TypeZone =="Communes") %>%
      dplyr::mutate(seuil_pop = dplyr::case_when(
        .data$population_municipale > pop_3  ~ glue::glue("communes de plus\nde ",pop_3," habitants"),
        .data$population_municipale > pop_2  ~ glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
        .data$population_municipale > pop_1  ~ glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
        .data$population_municipale > 0  ~ glue::glue("communes de moins\nde ",pop_1," habitants"),
        TRUE ~ "")
        ) %>%
      dplyr::mutate(seuil_code = dplyr::case_when(
        .data$population_municipale > pop_3  ~ "D",
        .data$population_municipale > pop_2  ~ "C",
        .data$population_municipale > pop_1  ~ "B",
        .data$population_municipale > 0  ~ "A",
        TRUE ~ "")
        ) %>%
      dplyr::rename ("population_n"="population_municipale") %>%
      dplyr::select (-.data$date)

    # table des seuils
    seuil_population <- population %>%
      dplyr::select (.data$CodeZone,.data$seuil_pop)

    # population du millesime
    population_n <- population %>%
      dplyr::select (.data$seuil_pop,.data$seuil_code, .data$population_n) %>%
      dplyr::group_by(.data$seuil_pop,.data$seuil_code) %>%
      dplyr::summarise(population_n = sum(.data$population_n, na.rm = T)) %>%
      dplyr::ungroup()

    # preparation des donnees ocsge
    ocsge2 <- ocsge %>%
      dplyr::mutate(
        date = lubridate::year(lubridate::as_date(.data$date)),
        espace_naturel = .data$autre_surface_naturelle + .data$surface_en_eau + .data$surface_naturelle_boisee
        ) %>%
      COGiter::filtrer_cog(reg = code_reg) %>%
      dplyr::filter(.data$date == millesime_ocsge,
                    .data$TypeZone %in% c("Communes")) %>%
      dplyr::select(
        .data$TypeZone, .data$Zone, .data$CodeZone,
        .data$espace_naturel,
        .data$espace_agricole, .data$espace_artificialise
        )

    ocsge3 <- ocsge2 %>%
      dplyr::left_join(seuil_population) %>%
      dplyr::group_by(.data$seuil_pop) %>%
      dplyr::summarise(espace_naturel = sum(.data$espace_naturel, na.rm = T),
                       espace_agricole = sum(.data$espace_agricole, na.rm = T),
                       espace_artificialise = sum(.data$espace_artificialise, na.rm = T))

    # preparation graphiques
    data <- ocsge3 %>%
      tidyr::pivot_longer(.data$espace_naturel:.data$espace_artificialise,
                          names_to = "variable",
                          values_to = "valeur"
                          ) %>%
      dplyr::group_by(.data$seuil_pop) %>%
      dplyr::mutate(variable = .data$variable,
                    valeur = .data$valeur / 10000,
                    taux = .data$valeur / sum(.data$valeur, na.rm = T) * 100
                    ) %>%
      dplyr::ungroup() %>%
      dplyr::mutate(Zone = factor(.data$seuil_pop, levels = c(glue::glue("communes de moins\nde ",pop_1," habitants"),
                                                                   glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
                                                                   glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
                                                                   glue::glue("communes de plus\nde ",pop_3," habitants"))) %>%
                      forcats::fct_rev()) %>%
      dplyr::mutate(variable = forcats::fct_relevel(variable,"espace_naturel","espace_agricole","espace_artificialise"))



    # graphique pourcentages
    graph_1_6 <- data %>%
      ggplot2::ggplot() +
      ggplot2::aes(x = .data$Zone, y = .data$taux, ymin = 0, ymax = .data$taux, group = .data$variable, color = .data$variable) +
      ggplot2::geom_linerange(size = 1.1, position = ggplot2::position_dodge(width = 0.5)) +
      ggplot2::geom_point(size = 1.5, position = ggplot2::position_dodge(width = 0.5)) +
      ggplot2::aes(label = mapfactory::format_fr_pct_2(.data$taux)) +
      ggplot2::geom_label(
        position = ggplot2::position_dodge(width = 0.5),
        vjust = 0.5, hjust = -0.5, alpha = .7
        ) +
      ggplot2::coord_flip() +
      ggplot2::scale_y_continuous(
        limits = c(0, max(data$taux) * 1.2),
        labels = scales::label_number(big.mark = " ", decimal.mark = ",", suffix = " %"),
        expand = c(0, 0)
        ) +
      ggplot2::labs(
        title = "R\u00e9partition des surfaces entre <span style = 'color:#FF8D7E'>espaces artificialis\u00e9s </span>, <br> <span style = 'color:#E3CE00'>espaces agricoles</span> et <span style = 'color:#169B62'>espaces naturels </span>",
        subtitle = glue::glue("en {millesime_ocsge} (en %)"),
        caption = glue::glue("Source : OCSGE {millesime_ocsge}"),
        x = NULL, y = NULL
        ) +
      ggplot2::theme(
        panel.grid.major.y = ggplot2::element_blank(),
        legend.position = "none",
        plot.title = ggtext::element_textbox_simple(size = 14, lineheight = 1, hjust = 0),
        plot.subtitle = ggplot2::element_text(size = 12),
        plot.caption = ggplot2::element_text(size = 10)
        ) +
      ggplot2::scale_color_manual(values = c("#169B62","#E3CE00","#FF8D7E" ))

    return(graph_1_6)
  }
  if (code_reg  %notin% c('52')) {
    return(invisible(NULL))
  }
}
