#' Creation du graphique de la part des surfaces artificialisees (Teruti-Lucas)
#' @description Graphique en barres de la part des surfaces artificialisees dans la surface departementale et regionale en % (Teruti-Lucas)
#'
#' @param millesime_teruti une année parmi les millesimes sélectionnables par l'utilisateur, au format numerique.
#' @param code_reg code insee de la région sur laquelle construire le graphique
#'
#' @return Un diagramme en barres
#'
#' @importFrom dplyr filter select mutate group_by desc arrange
#' @importFrom COGiter filtrer_cog
#' @importFrom forcats fct_inorder
#' @importFrom ggplot2 ggplot aes geom_bar labs scale_y_continuous theme geom_text geom_col scale_fill_manual
#' @importFrom scales number_format
#' @importFrom glue glue
#' @importFrom lubridate make_date
#' @importFrom tidyr spread gather
#' @importFrom tricky set_standard_names
#' @importFrom mapfactory format_fr
#' @importFrom ggtext element_markdown
#' @importFrom COGiter filtrer_cog
#' @importFrom attempt stop_if stop_if_not
#'
#' @export
#'
#' @examples
#' creer_graphe_1_5(millesime_teruti=2018,code_reg = '52')
creer_graphe_1_5 <- function(millesime_teruti,code_reg){
  couleur_hors_voirie <- "#FF8D7E"
  couleur_voirie <- "#82534b"

  attempt::stop_if(millesime_teruti, is.null, msg = "millesime_teruti n'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_teruti, is.numeric, msg = "millesime_teruti n'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n'est pas un code r\u00e9gion valide")

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # Creation de la table utile a la production du graphique
  data <- teruti %>%
    dplyr::mutate(valeur=as.numeric(.data$valeur)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$date == lubridate::make_date(millesime_teruti,"01","01")) %>%
    tidyr::spread(key=.data$variable,value=.data$valeur,fill=0) %>%
    tricky::set_standard_names() %>%
    dplyr::arrange(.data$typezone) %>%
    dplyr::mutate(zone = forcats::fct_drop(.data$zone) %>% forcats::fct_inorder(),
                   voiries=(.data$sols_revetus )/.data$tous_sols*100,
                   hors_voiries=(.data$sols_batis+ .data$sols_stabilises+ .data$autres_sols_artificialises )/.data$tous_sols*100
    ) %>%
    dplyr::select(.data$typezone,.data$codezone,.data$zone,.data$voiries,.data$hors_voiries)%>%
    tidyr::gather(variable,valeur,.data$voiries:.data$hors_voiries)%>%
    dplyr::mutate(variable = replace(.data$variable, .data$variable=="hors_voiries","surfaces artificialis\u00e9es hors voiries"),
                  codezone = replace(.data$codezone, .data$codezone==code_reg,"R\u00e9gion"))%>%
    dplyr::mutate(variable=factor(.data$variable,levels=c("surfaces artificialis\u00e9es hors voiries","voiries"))%>% forcats::fct_inorder()) %>%
    dplyr::group_by(.data$typezone,.data$codezone,.data$zone) %>%
    dplyr::arrange(.data$codezone, dplyr::desc(.data$variable)) %>%
    dplyr::mutate(position = cumsum(.data$valeur) - 0.5 * .data$valeur)

  graph_1_5<-data  %>%
    ggplot2::ggplot(ggplot2::aes(x=.data$zone,y=.data$valeur)) +
    ggplot2::geom_col(ggplot2::aes(fill = .data$variable), width = 0.9)+
    ggplot2::geom_text(ggplot2::aes(y = .data$position, label = mapfactory::format_fr(.data$valeur,1,pourcent = TRUE), group =.data$variable), color = "white", size=3)+
    ggplot2::labs(title= glue::glue("Part des surfaces artificialis\u00e9es pour les d\u00e9partements\net la r\u00e9gion en {millesime_teruti} (en %)"),
                  subtitle=glue::glue("<span style = 'color:{couleur_hors_voirie}'> Hors voiries</span> et <span style = 'color:{couleur_voirie}'> en voiries</span>"),
                  x="",
                  y="",
                  fill="",
                  caption = glue::glue("Source : Teruti-Lucas {millesime_teruti}"))+
    ggplot2::theme(legend.position = "none",
                   plot.subtitle = ggtext::element_markdown(size = 12, lineheight = 1.2))+
    ggplot2::scale_y_continuous(labels = scales::number_format(suffix = " %", accuracy = 1)) +
    ggplot2::coord_flip() +
    ggplot2::scale_x_discrete(limits=rev) +
    ggplot2::scale_fill_manual(values = c(couleur_voirie,couleur_hors_voirie))

  return(graph_1_5)

}
