#' Creation du graphique du taux d'artificialisation par region (Teruti-Lucas)
#' @description Creation du graphique representant le taux d'artificialisation par region (Teruti-Lucas)
#'
#' @param millesime_teruti une année parmi les millesimes sélectionnables par l'utilisateur, au format numerique.
#' @param code_reg code insee de la région sur laquelle construire le graphique
#'
#' @return Un diagramme en barres
#'
#' @importFrom dplyr filter select group_by mutate first ungroup pull
#' @importFrom forcats fct_drop fct_reorder
#' @importFrom ggiraph geom_point_interactive ggiraph
#' @importFrom ggplot2 ggplot aes geom_bar labs scale_y_continuous theme geom_text element_text scale_fill_manual scale_color_manual
#' @importFrom scales number_format
#' @importFrom glue glue
#' @importFrom attempt stop_if stop_if_not
#' @importFrom mapfactory format_fr_pct_2
#'
#' @export
#'
#' @examples
#' creer_graphe_1_1(millesime_teruti=2018 , code_reg = "52")
creer_graphe_1_1 <- function(millesime_teruti=NULL,
                             code_reg = NULL){

  attempt::stop_if(millesime_teruti, is.null, msg = "millesime_teruti n'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_teruti, is.numeric, msg = "millesime_teruti n'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n'est pas un code r\u00e9gion valide")

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # Creation de la table utile a la production du graphique
  data_france <- teruti %>%
    dplyr::mutate(valeur=as.numeric(.data$valeur)) %>%
    dplyr::filter(.data$CodeZone == "FRMETRO",
                  .data$date == lubridate::make_date(millesime_teruti,"01","01")) %>%
    tidyr::spread(key=.data$variable,value=.data$valeur,fill=0) %>%
    tricky::set_standard_names() %>%
    dplyr::mutate (taux_artificialisation=.data$sols_artificiels/.data$tous_sols*100)
  surf_artificialise_france <- data_france %>%
    dplyr::pull(.data$sols_artificiels)

  data <- teruti %>%
    dplyr::mutate(valeur=as.numeric(.data$valeur)) %>%
    # Filtres pour ne garder que les donnees de la region et du millesime sélectionne
    dplyr::filter(.data$TypeZone == "R\u00e9gions",
                  !(.data$CodeZone %in% c("01","02","03","04","06")),
                  .data$date == lubridate::make_date(millesime_teruti,"01","01")) %>%
    tidyr::spread(key=.data$variable,value=.data$valeur,fill=0) %>%
    tricky::set_standard_names() %>%
    dplyr::mutate (taux_artificialisation=.data$sols_artificiels/.data$tous_sols*100,
                   part_dans_surface_nationale=.data$sols_artificiels/surf_artificialise_france*100,
                   couleur_barre = dplyr::case_when(
                     .data$codezone==code_reg~ 1,
                     TRUE ~ 0)
    ) %>%
    dplyr::select(.data$zone,.data$taux_artificialisation,.data$part_dans_surface_nationale,.data$couleur_barre)

  valeur_max <- max(data$taux_artificialisation,na.rm=T)

  graph_1_1<-data  %>%
    ggplot2::ggplot(ggplot2::aes(x=forcats::fct_reorder(.data$zone,.data$taux_artificialisation,.desc=F),
                                 ymax=valeur_max+4,
                                 y=.data$taux_artificialisation,
                                 fill=as.factor(.data$couleur_barre))) +
    ggplot2::geom_bar(stat="identity")+
    ggplot2::geom_text(ggplot2::aes(y=.data$taux_artificialisation ,
                                    label=mapfactory::format_fr_pct_2(.data$taux_artificialisation,accuracy = 0.1),
                                    color = as.factor(.data$couleur_barre)
                                    ),
                       position= ggplot2::position_dodge(width=1),
                       vjust=0.5,
                       hjust=-0.5,
                       size=3) +
    ggplot2::coord_flip(expand = FALSE) +
    ggplot2::labs(title=glue::glue("Taux d'artificialisation par r\u00e9gion en {millesime_teruti}"),subtitle="",x="",y="",
                  fill="",
                  caption = glue::glue("Source : Teruti-Lucas {millesime_teruti}"))+
    ggplot2::theme(axis.text.x = ggplot2::element_text(), legend.position = "none") +
    ggplot2::scale_y_continuous(labels = scales::number_format(suffix = " %", accuracy = 1)) +
    ggplot2::scale_fill_manual(values = c("#FF8D7E","#82534b")) +
    ggplot2::scale_color_manual(values = c("#FF8D7E","#82534b"))

  return(graph_1_1)

 }
