#' Creation de la carte de l evolution de l artificialisation sur 10 ans en ha par departement
#' @description Carte de l evolution de l artificialisation sur 10 ans en ha par departement
#'
#' @param millesime_obs_artif une année parmi les millesimes sélectionnables par l'utilisateur, au format numerique.
#'
#' @return Une carte
#'
#' @importFrom dplyr filter select mutate arrange group_by summarise left_join
#' @importFrom ggplot2 ggplot geom_sf stat_sf_coordinates aes coord_sf theme element_rect scale_size_area labs
#' @importFrom gouvdown scale_fill_gouv_continuous
#' @importFrom lubridate year
#' @importFrom glue glue
#' @importFrom sf read_sf st_bbox st_buffer st_crs
#' @importFrom attempt stop_if stop_if_not
#' @importFrom mapfactory format_fr
#'
#' @export
#'
#' @examples
#' creer_carte_2_2(millesime_obs_artif = 2020)
creer_carte_2_2 <- function(millesime_obs_artif) {
  attempt::stop_if(millesime_obs_artif, is.null, msg = "millesime_obs_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_obs_artif, is.numeric, msg = "millesime_obs_artif n\'est pas un nombre")

  # calcul des millesimes extremes du graphique
  millesime_debut <- millesime_obs_artif - 10
  millesime_fin <- millesime_obs_artif - 1

  # Creation de la table utile a la production du graphique
  data <- observatoire_artificialisation %>%
    dplyr::filter(.data$TypeZone == "D\u00e9partements") %>%
    dplyr::filter(!(.data$CodeZone %in% c("971", "972", "973", "974", "975", "976"))) %>%
    dplyr::select(.data$CodeZone, .data$TypeZone, .data$Zone, .data$date, .data$flux_naf_artificialisation_total) %>%
    dplyr::mutate(flux_naf_artificialisation_total = .data$flux_naf_artificialisation_total / 10000) %>%
    dplyr::mutate(date = as.character(lubridate::year(.data$date - 1))) %>%
    dplyr::filter(.data$date < millesime_obs_artif, .data$date > millesime_obs_artif - 11) %>% # conserve les 10 derniers millesimes
    dplyr::arrange(.data$Zone) %>%
    dplyr::group_by(.data$CodeZone, .data$TypeZone, .data$Zone) %>%
    dplyr::summarise(`total en hectares` = sum(.data$flux_naf_artificialisation_total, na.rm = T)) %>%
    dplyr::right_join(COGiter::departements_geo,., by = c("DEP" = "CodeZone"))

  # creation de la carte
  monde_file <- system.file("maps", "countries_voisins-10m.gpkg", package = "mapfactory")
  monde <- sf::read_sf(monde_file) %>%
    dplyr::select(.data$name)

  bbox_reg <- sf::st_bbox(sf::st_buffer(data, 50000))

  y_min <- bbox_reg$ymin
  y_max <- bbox_reg$ymax
  x_min <- bbox_reg$xmin
  x_max <- bbox_reg$xmax

  carte_2_2 <- ggplot2::ggplot(data) +
    ggplot2::geom_sf(data = monde, fill = "light grey") +
    # carte modifiee
    # ggplot2::geom_sf(ggplot2::aes(fill= .data$`total en hectares`),
    #                  color = "black")+
    ggplot2::geom_sf(ggplot2::aes(fill= .data$`total en hectares`))+
      ggplot2::coord_sf(
    xlim = c(x_min, x_max),
    ylim = c(y_min, y_max),
    expand = FALSE,
    crs = sf::st_crs(data),
    datum = NA
  ) +


    # ggplot2::geom_sf(fill = "white") +
    # ggplot2::stat_sf_coordinates(ggplot2::aes(
    #   size = .data$`total en hectares`,
    #   fill = .data$`total en hectares`
    # ),
    # color = "black",
    # shape = 21,
    # alpha = 0.8
    # ) +
    # ggplot2::coord_sf(
    #   xlim = c(x_min, x_max),
    #   ylim = c(y_min, y_max),
    #   expand = FALSE,
    #   crs = sf::st_crs(data),
    #   datum = NA
    # ) +
    ggplot2::theme(
      panel.background = ggplot2::element_rect(fill = "light blue")
    ) +
    gouvdown::scale_fill_gouv_continuous(palette = "pal_gouv_h", reverse = TRUE, name = "", labels = function(x) mapfactory::format_fr(x, dec = 0)) +
    # ggplot2::scale_size_area(name="")+
    ggplot2::scale_size_continuous(range = c(0, 20), labels = function(x) mapfactory::format_fr(x, dec = 0)) +
    ggplot2::labs(
      title = glue::glue("Nouvelles surfaces consomm\u00e9es entre {millesime_debut} et {millesime_fin} (10 ans)"),
      subtitle = "",
      y = "",
      x = "",
      caption = glue::glue("Source : Observatoire de l\'artificialisation {millesime_obs_artif}")
    )



  return(carte_2_2)
}
