#' Import d'un dataset du serveur
#'
#' @param con Le connecteur de classe PosgreSQLConnection.
#' @param schema Le nom du schema sur le serveur de données.
#' @param table Le nom de la table sur le serveur de données.
#'
#' @importFrom attempt stop_if
#' @importFrom DBI dbReadTable
#' @importFrom RPostgreSQL postgresqlpqExec
#' @return Un dataframe.
#' @export

get_data <- function(con = NULL,
                     schema = NULL,
                     table = NULL) {
  attempt::stop_if(con, is.null, msg = "con n'est pas renseign\u00e9")
  attempt::stop_if(schema, is.null, msg = "schema n'est pas renseign\u00e9")
  attempt::stop_if(table, is.null, msg = "table n'est pas renseign\u00e9")

  data <- DBI::dbReadTable(con, c(schema,table))
  return(data)
}




#' import d'un dataset du serveur
#'
#' @param con Le connecteur de classe PosgreSQLConnection.
#' @param schema Le nom du schema sur le serveur de données.
#' @param table Le nom de la table sur le serveur de données.
#' @importFrom attempt stop_if
#' @importFrom DBI dbSendQuery dbReadTable Id
#'
#' @return un dataframe
#' @export

get_data_dbi <- function(con = NULL,
                         schema = NULL,
                         table = NULL) {
  attempt::stop_if(con, is.null, msg = "con n'est pas renseign\u00e9")
  attempt::stop_if(schema, is.null, msg = "schema n'est pas renseign\u00e9")
  attempt::stop_if(table, is.null, msg = "table n'est pas renseign\u00e9")
  tbl <- DBI::Id(schema = schema, table = table)
  data <- DBI::dbReadTable(con, tbl)
  return(data)
}


#' Import d'une table a partir du serveur de donnees, fonction qui s'adapte aux types de table a charger : recuperation des modalites des facteurs si chargee avec {rpostgis}, sf dataframe si table geographique.
#'
#' @param table Le nom de la table presente sur le serveur de données.
#' @param schema Le nom du schema contenant la table a rapatrier.
#' @param db Le nom de la base de donnees sur laquelle se connecter.
#' @param user L'identifiant de l'utilisateur qui se connecte.
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron
#'
#' @return la table de donnees, au format dataframe ou sf dataframe si une colonne de type 'USER-DEFINED' est presente dans la table sgbd
#' @importFrom attempt stop_if stop_if_not
#' @importFrom DBI dbExistsTable dbReadTable dbDisconnect
#' @importFrom dplyr filter arrange select all_of relocate rename_with
#' @importFrom glue glue
#' @importFrom rlang .data
#' @importFrom rpostgis dbTableInfo
#' @importFrom sf st_read st_drop_geometry st_geometry
#' @importFrom tibble column_to_rownames
#' @export
#'
#' @examples
#' \dontrun{
#' importer_data(table = "test_iris", schema = "public", db = "public", user = "does")
#' }
#' @source {rpostgis} http://mablab.org/rpostgis/


importer_data <- function(table = NULL, schema = NULL, db  = "public", server = NULL, user = "does") {
  attempt::stop_if(table, is.null, msg = "l'argument table n'est pas renseign\u00e9")
  attempt::stop_if(schema, is.null, msg = "l'argument schema n'est pas renseign\u00e9")
  attempt::stop_if(db, is.null, msg = "l'argument db n'est pas renseign\u00e9")
  attempt::stop_if(user, is.null, msg = "l'argument user n'est pas renseign\u00e9")

  con <- connect_to_db(db = db, user = user, server = server)
  name <- c(schema, table)

  attempt::stop_if_not(DBI::dbExistsTable(conn = con, name = name),
                       msg = paste0("La table '", paste(name, collapse = "'.'"), "' est introuvable."))

  # on recupere la liste des noms de champs de la table et leurs caracteristiques
  champs <-  rpostgis::dbTableInfo(conn = con,  name = c(schema, table), allinfo = FALSE)

  # recupere-t-on des proprietes de dataframe R dans zz_r_df_def ?
  if(DBI::dbExistsTable(conn = con, name = c(schema, "zz_r_df_def"))) {
    def_table <- DBI::dbReadTable(con, c(schema, "zz_r_df_def")) %>%
      dplyr::filter(.data$table_nm == table)
    if(nrow(def_table)==0) {
      message(paste0("Pas de renseignements trouv\u00e9 sur ", table,
                     " dans la table des propri\u00e9t\u00e9s des dataframes R, un import standard sera r\u00e9alis\u00e9. "))
      defs = data.frame()
    } else {
      sql_query <- paste0("SELECT unnest(df_def[1:1]) as nms, unnest(df_def[2:2]) as defs, unnest(df_def[3:3]) as atts, unnest(df_def[4:4]) as r_nms FROM ",
                          schema, ".\"zz_r_df_def\" WHERE table_nm = '", table, "';")
      defs <- DBI::dbGetQuery(conn = con, sql_query)
    }
  } else {
    message("La table de propri\u00e9t\u00e9s des dataframes R est introuvable, un import standard sera r\u00e9alis\u00e9. ")
    def_table = data.frame()
    defs = data.frame()
  }

  # on distingue 2 cas : table geo, table de donnees
  if(any(grepl("USER-DEFINED", champs$data_type))) { # la signature des champs de geometry
    data_sf <- sf::st_read(dsn = con, layer = c(schema, table))

    if ("id_row__" %in% champs$column_name) { # on enleve le champ de cle primaire eventuellement cree après avoir remis dans l'ordre les enregistrements
      data_sf <- dplyr::arrange(.data = data_sf, .data$id_row__) %>%
        dplyr::select(-.data$id_row__)
    }
    data <- sf::st_drop_geometry(data_sf)

  } else {
    data <- DBI::dbReadTable(con, c(schema, table))

    if ("id_row__" %in% champs$column_name) {
      data <- dplyr::arrange(.data = data, .data$id_row__) %>%
        dplyr::select(-.data$id_row__)
    }
  }

  # repasser les éventuelle noms de lignes en noms de lignes de dataset
  if ("r_row_names__" %in% champs$column_name) {
    data <- tibble::column_to_rownames(.data = data, var = "r_row_names__")
    defs <- dplyr::filter(defs, .data$nms != "r_row_names__")
  }

  # récupérer les autres propriétés initiales du dataset
  if(nrow(def_table) > 0) {

    ## assign types  i <- names(data)[2]
    for (i in names(data)) {
      att <- defs[defs$nms == i, ]
      if (length(att[, 1]) == 0) {
        next
      }
      if (!is.na(att$atts)) {
        ## begin modular handling of different data type attributes
        ## (add new below) handle factors
        if (att$defs %in% c("factor", "ordered")) {
          levs <- unlist(strsplit(att$atts, "/*/", fixed = TRUE))
          ordered <- ifelse(att$defs == "ordered", TRUE, FALSE)
          # une précaution pour s'assurer que les levels ne presentent pas d'anomalie d'encodage ou autre par rapport au donnees chargees
          if(all(data[, i] %in% c(levs, NA, ""))) {
            data[, i] <- factor(as.character(data[, i]), levels = levs[levs != ""], ordered = ordered)
          } else {
            message(paste0("Les levels du facteur ", i, " sont incompatibles avec les donnees en base, cette variable est importee avec le type character.\n"))
          }

        }
        ## handle POSIX time zones
        if (att$defs %in% c("POSIXct", "POSIXt")) {
          data[, i] <- list(eval(parse(text = paste0("as.", att$defs, "(as.character(data[,i]), tz='",
                                                  Sys.timezone(), "')"))))
          ## assign R tz
          eval(parse(text = paste0("attributes(data$", i,
                                   ")$tzone <- att$atts")))
        }
        if (att$defs == "POSIXlt") {
          data[, i] <- list(eval(parse(text = paste0("as.", att$defs, "(as.character(data[,i]),
                                          tz=att$atts)"))))
        }
        ## end modular handling of different data types
      } else {
        data[, i] <- do.call(paste0("as.", att$defs), list(data[, i]))
      }
    }
  }

  # remettre le champ geometrie (au bon endroit)
  if(exists("data_sf") & nrow(def_table) > 0) {
    sf::st_geometry(data) <- sf::st_geometry(data_sf)
    # renommage du champ geo a l'identique
    sf_col_name <- attributes(data_sf)$sf_column
    data <- dplyr::rename_with(data, ~gsub("^geometry$", sf_col_name, .x))
    sf::st_geometry(data) <- sf_col_name

    data <- dplyr::relocate(data, dplyr::all_of(defs$nms))
  }

  if(exists("data_sf") & nrow(def_table) == 0) {
    data <- data_sf
  }

# renommer avec les éventuels noms initiaux
  if(nrow(def_table) > 0) {
    data <- dplyr::relocate(data, dplyr::all_of(defs$nms))
    names(data) <- defs$r_nms
  }

  DBI::dbDisconnect(conn = con)
  rm(con)
  return(data)
}

# ecln_cogifie2 <- importer_data(table = "test_ecln_cog", schema = "public", db = "public", server = "localhost", user = "does")
# iris2 <- importer_data(table = "test_iris", schema = "public", db = "public", server = "localhost", user = "does") # pk = NULL, post_row_name = TRUE, overwrite = TRUE
# all(iris == iris2)
# dep_geo <- importer_data(table = "test_dep", schema = "public", db = "public", server = "localhost", user = "does") # pk = "DEP", post_row_name = FALSE, overwrite = TRUE
# test_epci <- COGiter::epci %>% dplyr::select(-ends_with("DE_L_EPCI")) %>% dplyr::slice(1:270) %>% dplyr::mutate_if(is.factor, forcats::fct_drop)
# epci_test <- importer_data(table = "test_epci", schema = "public", db = "datamart", user = "does")
# all(epci_test == test_epci)
