#' commenter_table : ajouter (ou remplacer) un commentaire de table dans le SGBD
#'
#' @param comment : Le texte du commentaire a poster.
#' @param table : Le texte du commentaire a poster.
#' @param schema : Le nom du schema qui contient la table a commenter.
#' @param db : Le nom de la base qui contient la table a commenter.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#'
#' @importFrom attempt stop_if stop_if_not
#' @importFrom DBI dbDisconnect
#' @importFrom glue glue
#' @importFrom rpostgis dbComment
#' @return NULL
#' @export
#'
#' @examples
#' \dontrun{
#' commenter_table(comment = "Je suis un commentaire important.", table = "test_iris",
#'                 schema = "public", db  = "public", user = "does")
#' }

commenter_table <- function(comment = NULL, table = NULL, schema = NULL, db  = "public", user = "does") {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db, user)

  # verification de la validite des arguments
  attempt::stop_if(comment, is.null, msg = "L'argument comment n\'est pas renseign\u00e9. ")
  attempt::stop_if(schema, is.null, msg = "L'argument schema n\'est pas renseign\u00e9. ")
  attempt::stop_if(table, is.null, msg = "L'argument table n\'est pas renseign\u00e9. ")
  attempt::stop_if_not(table %in% list_tables(con, schema), msg = glue::glue("Il n'y a pas de table {table} dans le schema {schema}. "))

  meta <- paste0("\nCommentaire post\u00e9 le ", format(Sys.Date(), '%d/%m/%Y'), ", par ", Sys.getenv("USERNAME"), ".")

  # securisation encodage des accents et caracteres speciaux
  comment2 <- paste0(enc2utf8(comment), enc2utf8(meta))

  rpostgis::dbComment(conn = con, name = c(schema, table), comment = comment2, type = "table", exec = TRUE)

  DBI::dbDisconnect(con)

  return(invisible(NULL))
}



#' commenter_schema : ajouter (ou remplacer) un commentaire pour un schema du SGBD
#'
#' @param comment : le texte du commentaire a poster
#' @param schema : le nom du schema a commenter
#' @param db : le nom de la base qui contient le schema a commenter
#' @param user : l'identifiant de l'utilisateur SGBD qui se connecte
#' @importFrom attempt stop_if
#' @importFrom DBI dbDisconnect
#' @importFrom rpostgis dbComment
#' @return NULL
#' @export
commenter_schema <- function(comment = NULL, schema = NULL, db  = "public", user = "does") {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db, user)

  # verification de la validite des arguments
  attempt::stop_if(comment, is.null, msg = "L'argument comment n\'est pas renseign\u00e9. ")
  attempt::stop_if(schema, is.null, msg = "L'argument schema n\'est pas renseign\u00e9. ")

  comment2 <- enc2utf8(comment)

  rpostgis::dbComment(conn = con, name = schema, comment = comment2, type = "schema", exec = TRUE)

  DBI::dbDisconnect(con)

  return(invisible(NULL))
}

#' get_schema_comment : Recuperer les commentaires d'un schema sur le SGBD
#'
#' @param schema : Le nom du schema  sur le SGBD dont on veut recuperer la description
#' @param db : Le nom de la base qui contient le schema dont on recherche le descriptif.
#' @param user : l'identifiant de l'utilisateur SGBD qui se connecte
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron
#' @return le commentaire recherche dans un dataframe de 3 champs (base, nom_schema, commentaire)
#'
#' @export
#' @importFrom attempt stop_if_not stop_if
#' @importFrom DBI dbGetQuery
#' @importFrom dplyr transmute
#' @importFrom rlang .data


get_schema_comment <- function(schema = NULL, db = "public", user = "does", server = NULL) {
  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db = db, user = user, server = server)

  # verification de la validite des arguments
  attempt::stop_if(schema, is.null, msg = "L\'argument schema n\'est pas renseign\u00e9. ")
  attempt::stop_if_not(schema %in% list_schemas(con), msg = glue::glue("Il n\'y a pas de schema {schema} dans la base de donn\u00e9es. "))

  # construction de la requete sql
  query <-  paste0("SELECT obj_description('", schema, "'::regnamespace) AS commentaire;")

  # recupération du commentaire du schema
  com <- DBI::dbGetQuery(conn = con, statement = query ) %>%
    dplyr::transmute(nom_schema = schema, commentaire = .data$commentaire)

  DBI::dbDisconnect(con)
  rm(con)

  return(com)
}
