
#' commenter_champs : ajouter (ou remplacer) un commentaire de champ d'une table dans le SGBD
#'
#' @param comment : Le texte du commentaire a poster.
#' @param var : Le nom du champ a commenter.
#' @param table : La table SGBD dont on veut decrire un champ.
#' @param schema : Le nom du schema qui contient la table a commenter.
#' @param db : Le nom de la base qui contient la table a commenter.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron
#'
#' @importFrom attempt stop_if stop_if_not
#' @importFrom DBI dbSendQuery dbDisconnect
#' @importFrom glue glue
#' @importFrom rpostgis dbTableInfo
#' @return NULL
#' @export
#'
#' @examples
#' \dontrun{
#' commenter_champs(comment = "Je suis un commentaire ultra important sur la variable 'Species'.",
#'                  var = "Species", table = "test_iris",
#'                  schema = "public", db = "production", user = "does")
#' }

commenter_champs <- function(comment = NULL, var = NULL, table = NULL, schema = NULL, db = "production", user = "does", server = NULL) {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db = db, user = user, server = server)

  # verification de la validite des arguments
  attempt::stop_if(comment, is.null, msg = "L\'argument comment n\'est pas renseign\u00e9. ")
  attempt::stop_if(var, is.null, msg = "L\'argument var n\'est pas renseign\u00e9. ")
  attempt::stop_if(schema, is.null, msg = "L\'argument schema n\'est pas renseign\u00e9. ")
  attempt::stop_if(table, is.null, msg = "L\'argument table n\'est pas renseign\u00e9. ")
  attempt::stop_if_not(table %in% list_tables(con, schema), msg = glue::glue("Il n\'y a pas de table {table} dans le schema {schema}. "))
  attributs <- rpostgis::dbTableInfo(con, c(schema, table))$column_name
  attempt::stop_if_not(any(c(var, standard_var_pg(var)) %in% attributs), msg = glue::glue("Il n\'y a pas de champ {var} dans la table {table}. "))

  # normalisation du nom de champ au besoin
  if(!(var %in% attributs)) {
    var <- standard_var_pg(var)
    }
  # echappement des ' pour que la requete passe en entier et ajout guillemet sur le nom des variables pour gerer les noms de champs avec majuscules
  var2 <- paste0("\"", var, "\"")
  lib_var <- gsub("\'", "\'\'", comment)
  requete <- paste0("COMMENT ON COLUMN ", schema, ".", table, ".", var2, " IS \'", enc2utf8(lib_var), "\';")
  DBI::dbSendQuery(conn = con, statement = requete)

  DBI::dbDisconnect(con)
  rm(con)
  return(invisible(NULL))
}

#' Postage de commentaires sur tous ou certains champs d'une table SGBD a partir d'un dataframe dictionnaire d'attributs
#'
#' @param dico Le dataframe dictionnaire d'attributs, comprend deux champs le 1er pour le nom des variables, le second pour leur explicitation. Le nommage de ces deux champs n'a pas d'importance, leur ordre importe en revanche. Ce dataframe comprend autant de lignes que d'attributs que l'on souhaite documenter.
#' @param table Le nom de la table sur le SGBD dont on veut decrire les champs
#' @param schema : Le nom du schema qui contient la table a commenter.
#' @param db : Le nom de la base qui contient la table a commenter.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron
#'
#' @importFrom attempt stop_if_not stop_if
#' @importFrom DBI dbSendQuery dbDisconnect
#' @importFrom glue glue
#' @importFrom purrr map
#' @importFrom rpostgis dbTableInfo
#' @return NULL
#' @export
#'
#' @examples
#' \dontrun{
#' dico_iris <- data.frame(VAR = names(iris), VAR_LIB = paste("libell\u00e9 de", names(iris)))
#' post_dico_attr(dico = dico_iris, table = "test_iris", schema = "public", db = "public",
#'                user = "does", server = "localhost")
#' }
#'


post_dico_attr <- function(dico = NULL, table = NULL, schema = NULL, db = "production", user = "does", server = NULL) {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db = db, user = user, server = server)

  # verification de la validite des arguments
  attempt::stop_if_not(dico, is.data.frame, msg = "L\'argument dico ne correpond pas \u00e0 un dataframe. ")
  attempt::stop_if(nrow(dico) == 0, msg = "L\'argument dico est vide. ")
  attempt::stop_if_not(length(dico)==2, msg = "L\'argument dico doit comprendre 2 champs : le 1er pour le nom des variables, le second pour leur explicitation. ")
  attempt::stop_if(schema, is.null, msg = "L\'argument schema n\'est pas renseign\u00e9. ")
  attempt::stop_if(table, is.null, msg = "L\'argument table n\'est pas renseign\u00e9. ")
  attempt::stop_if_not(table %in% list_tables(con, schema), msg = glue::glue("Il n\'y a pas de table {table} dans le schema {schema}. "))
  attributs <- rpostgis::dbTableInfo(con, c(schema, table))$column_name
  attempt::stop_if_not(all(dico[[1]] %in% attributs) | all(standard_var_pg(dico[[1]]) %in% attributs),
                       msg = glue::glue("Les champs list\u00e9s dans 'dico' ne se retrouvent pas tous dans la table {table}. "))

  # Mise en forme du dico des variables
  dic_var <- dico
  names(dic_var) <- c("VAR", "VAR_LIB")
  # normalisation du nom de champ au besoin
  if(!(all(dico[[1]] %in% attributs))) { dic_var$VAR <- standard_var_pg(dic_var$VAR) }
  dic_var$VAR <- paste0("\"", dic_var$VAR, "\"")
  dic_var$VAR_LIB <- gsub("\'", "\'\'", dic_var$VAR_LIB)

  # une fonction de commentaire d'un seul champ qui n'ouvre pas de connexion
  comment_sql <- function(var) {
    # recuperation du libelle a partir du nom de la variable
    lib_var <- dic_var[dic_var$VAR ==  var, 2][[1]]
    # ecriture de la requete de commentaire du champ
    commentaire <- paste0("COMMENT ON COLUMN ", schema, ".", table, ".", var, " IS \'", enc2utf8(lib_var), "\';")
    # execution de la requete
    DBI::dbSendQuery(conn = con, statement = commentaire)
  }

  # Envoi des commentaires de champs en serie
  purrr::map(.x = dic_var$VAR, .f = ~ comment_sql(var = .x))

  DBI::dbDisconnect(con)
  rm(con)
  return(invisible(NULL))

}


#' get table comment : recuperer les commentaires d'une table presente sur le SGBD
#'
#' @param table Le nom de la table sur le SGBD dont on veut recuperer la description
#' @param schema : Le nom du schema qui contient la table dont on recherche le descriptif.
#' @param db : Le nom de la base qui contient la table dont on recherche le descriptif.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron
#'
#' @return le commentaire recherche dans un dataframe de 3 champs (nom_schema, nom_table, commentaire) et une ligne
#' @export
#' @importFrom attempt stop_if_not stop_if
#' @importFrom DBI dbGetQuery
#' @importFrom dplyr transmute
#' @importFrom rlang .data
#'

get_table_comment <- function(table = NULL, schema = NULL, db = "production", user = "does", server = NULL) {
  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db = db, user = user, server = server)

  # verification de la validite des arguments
  attempt::stop_if(schema, is.null, msg = "L\'argument schema n\'est pas renseign\u00e9. ")
  attempt::stop_if(table, is.null, msg = "L\'argument table n\'est pas renseign\u00e9. ")
  attempt::stop_if_not(table %in% list_tables(con, schema), msg = glue::glue("Il n\'y a pas de table {table} dans le schema {schema}. "))

  # construction de la requete sql
  sch.tb <- paste0(schema, ".", table)
  query <-  paste0("SELECT obj_description('", sch.tb, "'::regclass) AS commentaire;")

  # recupération du commentaire de table
  com <- DBI::dbGetQuery(conn = con, statement = query ) %>%
    dplyr::transmute(nom_schema = schema, nom_table = table, commentaire = .data$commentaire)

  DBI::dbDisconnect(con)
  rm(con)

  return(com)
}


#' get cols comment : recuperer les commentaires des champs d'une table presente sur le SGBD
#'
#' @param table Le nom de la table sur le SGBD dont on veut recuperer la description
#' @param schema : Le nom du schema qui contient la table dont on recherche le descriptif.
#' @param db : Le nom de la base qui contient la table dont on recherche le descriptif.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron.
#'
#' @return les commentaires recherches dans un dataframe de 4 champs (nom_schema, nom_table, nom_col, commentaire) et autant de lignes que de champs de la table commentes
#' @export
#' @importFrom attempt stop_if_not stop_if
#' @importFrom DBI dbGetQuery
#' @importFrom dplyr filter distinct
#' @importFrom rlang .data

get_cols_comment <- function(table = NULL, schema = NULL, db = "production", user = "does", server = NULL) {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db = db, user = user, server = server)

  # verification de la validite des arguments
  attempt::stop_if(table, is.null, msg = "L\'argument table n\'est pas renseign\u00e9. ")
  attempt::stop_if(schema, is.null, msg = "L\'argument schema n\'est pas renseign\u00e9. ")
  attempt::stop_if_not(table %in% list_tables(con, schema), msg = glue::glue("Il n\'y a pas de table {table} dans le schema {schema}. "))

  query <- paste0("SELECT table_schema as nom_schema, table_name as nom_table, column_name as nom_col, col_description((table_schema||'.'||table_name)::regclass::oid, ordinal_position) as commentaire ",
                  "FROM information_schema.columns WHERE table_schema = '", schema, "';")

  com <- DBI::dbGetQuery(conn = con, statement = query) %>%
    dplyr::filter(!is.na(.data$commentaire), .data$nom_table == table) %>%
    dplyr::distinct()

  DBI::dbDisconnect(con)
  rm(con)

  return(com)
}

#' Recuperer les commentaires d'une table et de ses champs sur le SGBD
#'
#' @param table Le nom de la table sur le SGBD dont on veut recuperer la description
#' @param schema : Le nom du schema qui contient la table dont on recherche le descriptif.
#' @param db : Le nom de la base qui contient la table dont on recherche le descriptif.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#' @param server L'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron.
#'
#' @return les commentaires recherches dans un dataframe de 4 champs (nom_schema, nom_table, commentaire) et autant de lignes que de champs de la table commentes
#' @export
#'
#' @examples
#' \dontrun{
#' get_table_comments(table = "test_iris", schema = "public", db = "public",
#'                    user = "does", server = "localhost")
#'}
get_table_comments <- function(table = NULL, schema = NULL, db = "production", user = "does", server = NULL) {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con <- connect_to_db(db = db, user = user, server = server)

  com <- dplyr::bind_rows(
    get_table_comment(table = table, schema = schema, db = db, user = user, server = server),
    get_cols_comment(table = table, schema = schema, db = db, user = user, server = server)
  )

  DBI::dbDisconnect(con)
  rm(con)

  return(com)
}


#' transferer les commentaires d'une table et de ses champs à une autre table identique sur le SGBD
#'
#' @param table_sce Le nom de la table sur le SGBD dont on veut recuperer la description
#' @param schema_sce : Le nom du schema qui contient la table dont on recherche le descriptif.
#' @param db_sce : Le nom de la base qui contient la table dont on recherche le descriptif.
#' @param table_dest Le nom de la table sur le SGBD que l'on souhaite decrire a partir de table_sce, ces deux tables doivent etre de structure identique
#' @param schema_dest : Le nom du schema qui contient la table que l'on souhaite decrire.
#' @param db_dest : Le nom de la base qui contient la table que l'on souhaite decrire.
#' @param user : L'identifiant de l'utilisateur SGBD qui se connecte.
#' @param server L'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron.
#'
#' @return NULL
#' @export
#'
#' @examples
#' \dontrun{
  #' transferer_table_comments(table_sce = "test_iris", schema_sce = "public", db_sce = "public",
  #' schema_dest = "tiger", db_dest = "public",
  #' user = "does", server = "localhost")
#' }
transferer_table_comments <- function(table_sce = NULL, schema_sce = NULL, db_sce = "public",
                                      table_dest = table_sce, schema_dest = NULL, db_dest = "public",
                                      user = "does", server = NULL) {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")
  con_dest <- connect_to_db(db = db_dest, user = user, server = server)

  # recuperation des commentaire de la table source
  tb_comment <- get_table_comments(table = table_sce, schema = schema_sce, db = db_sce, user = user, server = server)

  dico_atrbts <- tb_comment %>%
    dplyr::filter(!is.na(.data$nom_col)) %>%
    dplyr::select(.data$nom_col, .data$commentaire)

  # Envoi des commentaires sur la table de destination : commentaire de table, puis commentaires de champs.
  rpostgis::dbComment(conn = con_dest, name = c(schema_dest, table_dest), comment = tb_comment$commentaire[1], type = "table", exec = TRUE)
  if(nrow(dico_atrbts) >0) {
    post_dico_attr(dico = dico_atrbts, table = table_dest, schema = schema_dest, db = db_dest, user = user, server = server)
    # information de l'utilisateur :
    message("Dictionnaire des variables : ")
    print(dico_atrbts)
  }

  DBI::dbDisconnect(con_dest)
  rm(con_dest)

  return(invisible(NULL))
}




