#' Chargement d'un dataset sur le serveur
#'
#' @return La fonction charge le dataset sur le serveur.
#'
#' @param con Le connecteur de classe PosgreSQLConnection.
#' @param data Un dataframe a importer.
#' @param schema Le nom du schema sur le serveur de données.
#' @param table Le nom de la table sur le serveur de données.
#' @param overwrite TRUE si on veut ecraser le fichier deja present.
#'
#' @importFrom attempt stop_if stop_if_not message_if
#' @importFrom DBI dbWriteTable
#' @importFrom glue glue
#' @export

post_data <- function(con,
                      data = NULL,
                      schema = NULL,
                      table = NULL,
                      overwrite = FALSE
                      ) {
  attempt::stop_if(con, is.null, msg = "con n\'est pas renseign\u00e9")
  attempt::stop_if_not(deparse(substitute(con)), exists, msg = glue::glue("le connecteur {con} n\'existe pas"))
  attempt::stop_if(data, is.null, msg = "data n\'est pas renseign\u00e9")
  attempt::stop_if_not(data, is.data.frame, msg = glue::glue("le dataframe {data} n\'exsiste pas"))
  attempt::stop_if(schema, is.null, msg = "schema n\'est pas renseign\u00e9")
  attempt::stop_if(table, is.null, msg = "table n\'est pas renseign\u00e9")
  attempt::stop_if_not(schema, ~ .x %in% list_schemas(con), msg = glue::glue("le schema {schema} n\'existe pas"))
  attempt::stop_if(table %in% list_tables(con,schema) & !overwrite, msg = glue::glue("La table {table} existe d\u00e9j\u00e0 sur le schema {schema} et le param\u00e8 tre overwrite est \u00e0 FALSE"))
  attempt::message_if(table %in% list_tables(con,schema) & overwrite, msg = glue::glue("La table {table} est \u00e9cras\u00e9e sur le schema {schema}"))
  DBI::dbWriteTable(con, c(schema,table),  data, overwrite = overwrite )
  return(invisible(NULL))
}


#' Chargement d'un dataset sur le serveur en dbi
#'
#' @param con Le connecteur de classe PosgreSQLConnection.
#' @param data Un dataframe a importer.
#' @param schema Le nom du schema sur le serveur de donnees.
#' @param table Le nom de la table sur le serveur de donnees.
#' @param overwrite TRUE si on veut ecraser le fichier deja present.
#' @importFrom attempt stop_if stop_if_not
#' @importFrom DBI dbWriteTable Id dbSendQuery
#' @importFrom glue glue
#'
#' @return NULL, la fonction lance le chargement du dataset present dans l'environnement R et son ecriture sur le serveur
#' @export
#'

post_data_dbi <- function(con = NULL,
                          data = NULL,
                          schema = NULL,
                          table = NULL,
                          overwrite = FALSE) {
  attempt::stop_if(con, is.null, msg = "con n\'est pas renseign\u00e9")
  attempt::stop_if_not(deparse(substitute(con)), exists, msg = glue::glue("le connecteur {con} n\'existe pas"))
  attempt::stop_if(data, is.null, msg = "data n\'est pas renseign\u00e9")
  attempt::stop_if_not(data, is.data.frame, msg = glue::glue("le dataframe {data} n\'exsiste pas"))
  attempt::stop_if(schema, is.null, msg = "schema n\'est pas renseign\u00e9")
  attempt::stop_if(table, is.null, msg = "table n\'est pas renseign\u00e9")
  attempt::stop_if_not(schema, ~ .x %in% list_schemas(con), msg = glue::glue("le schema {schema} n\'existe pas"))
  attempt::stop_if(table %in% list_tables(con,schema) & !overwrite, msg = glue::glue("La table {table} existe d\u00e9j\u00e0 sur le schema {schema} et le param\u00e8 tre overwrite est \u00e0 FALSE"))
  DBI::dbSendQuery(con,"SET client_encoding TO \'windows-1252\'")
  tbl <- DBI::Id(schema = schema, table = table)
  DBI::dbWriteTable(con, tbl,  data, overwrite = overwrite )
  return(invisible(NULL))

}

#' Chargement d'un dataset sur le serveur avec rpostgis. Par rapport aux autres fonctions `post_data()`, `post_data_pg()` ouvre et ferme la connexion au SGBD, gere les dataframes spatiaux et les facteurs, et poste quelques metadonnees en commentaire de la table.
#'
#' @param data Un dataframe a verser sur le serveur de donnees.
#' @param table Le nom a donner a la table sur le serveur de donnees (character).
#' @param schema Le nom du schema sur le serveur de donnees.
#' @param pk Cle primaire a creer : indiquer soit le ou les nom(s) du champ qui doivent etre utilises (character vector), soit 1 pour utiliser la premiere colonne (numeric), laisser à NULL (choix par defaut) pour le creer a partir des numeros de lignes.
#' @param post_row_name Boleen indiquant si on souhaite verser le nom des lignes du dataframe, FALSE par defaut.
#' @param db La base sur laquelle se connecter.
#' @param user Le nom d'utilisateur pour se connecter au SGBD.
#' @param overwrite boleen : TRUE si on veut ecraser le fichier deja present, FALSE par defaut.
#' @param droits_schema boleen : TRUE (valeur par defaut) pour ajouter au niveau de la table versee les droits par defaut du schema.
#' @param server l'adresse ip du serveur, laisser à NULL pour utiliser le variable d'environnement du .Renviron
#' @importFrom attempt stop_if stop_if_not warn_if_not
#' @importFrom DBI dbExistsTable dbExecute dbGetQuery dbWriteTable dbDisconnect
#' @importFrom dplyr mutate across relocate
#' @importFrom forcats fct_relabel
#' @importFrom glue glue
#' @importFrom rlang .data
#' @importFrom rpostgis dbSchema dbAddKey dbComment pgInsert dbIndex
#' @importFrom sf st_write st_crs st_dimension
#' @importFrom tidyselect vars_select_helpers
#' @importFrom units drop_units
#'
#' @return NULL, la fonction verse le dataset de l'environnement R sur le serveur de donnees.
#' @export
#' @examples
#' \dontrun{
#' poster_data(data = iris, table = "test_iris", schema = "public", db = "public",
#'             pk = NULL, post_row_name = FALSE, overwrite = TRUE, user = "does")
#' }
#' @source {rpostgis} http://mablab.org/rpostgis/

poster_data <- function(data = NULL,
                        table = NULL,
                        schema = NULL,
                        pk = NULL,
                        post_row_name = FALSE,
                        db  = "public",
                        overwrite = FALSE,
                        droits_schema = TRUE,
                        user = "does",
                        server = NULL) {

  # ouverture de la connexion avec DBI::dbDriver("PostgreSQL")----------------
  con <- connect_to_db(db = db, user = user, server = server)

  # verification de la validite des arguments
  attempt::stop_if(data, is.null, msg = "'data' n\'est pas renseign\u00e9")
  attempt::stop_if_not(data, is.data.frame, msg = glue::glue("le dataframe 'data' n\'exsiste pas"))
  attempt::stop_if_not(any(pk[1]==1, is.null(pk), all(pk %in% names(data))), msg = glue::glue("mauvaise declaration de pk : doit etre 1 ou NULL ou un nom de variable de 'data' ou un vecteur de noms de variables de 'data'"))
  attempt::stop_if(schema, is.null, msg = "schema n\'est pas renseign\u00e9")
  attempt::stop_if(table, is.null, msg = "table n\'est pas renseign\u00e9")

  rpostgis::dbSchema(conn = con, name = schema, display = FALSE, exec = TRUE) # creation du schema au besoin
  attempt::warn_if_not(schema, ~ .x %in% list_schemas(con), msg = glue::glue("le schema {schema} n\'existe pas. Il va \u00eatre cr\u00e9\u00e9 sous {db}."))
  attempt::stop_if(table %in% list_tables(con = con, schema = schema, db = NULL, user = NULL) & !overwrite, msg = glue::glue("La table {table} existe d\u00e9j\u00e0 sur le schema {schema} et le param\u00e8tre overwrite est \u00e0 FALSE"))

  # passage des noms de champs en minuscule et standardisation----------------------
  noms_r <- names(data) # liste des noms de champs du jeu de donnees dans R
  noms_pg <- standard_var_pg(noms_r) # liste des noms corriges pour insertion dans le SGBD
  d <- data
  names(d) <- noms_pg # renommage

  if(post_row_name) {
    d <- dplyr::mutate(d, `r_row_names__` = attr(data, "row.names"))
    noms_r = c(noms_r, "row.names")
    noms_pg = names(d)
  }

  ## Creation de la table des definitions utiles pour le reimport dans R du jeu de donnee------------

  # Creation de la table (vide) et de sa structure dans le schema de destination
  if (!DBI::dbExistsTable(conn = con, c(schema, "zz_r_df_def"), table.only = TRUE)) {
    sql_query <- paste0("CREATE TABLE ", schema, ".\"zz_r_df_def\" (table_nm character varying, df_def text[]);")
    DBI::dbExecute(conn = con, sql_query)
    suppressMessages({
      rpostgis::dbAddKey(conn = con, c(schema, "zz_r_df_def"), colname = "table_nm",
                         type = "primary")
      rpostgis::dbComment(conn = con, c(schema, "zz_r_df_def"),
                          comment = "Table holding R dataframes columns definitions (for import/export using rpostgis :: db(Read/Write)DataFrame) or {datalibaba}.")
    })
    message("Nouvelle table de d\u00e9finition des R dataframes R cr\u00e9\u00e9e : ",
            schema, ".\"zz_r_df_def\".")
  }
  # Peuplement de la table des definitions
  ## >>> normaliser le type units tant que pas géré
  ## a <- units::deparse_unit(COGiter::departements_geo$AREA)
  ## dep_geo$AREA <- units::set_units(dep_geo$AREA, a)
  d <- units::drop_units(d)

  ## recuperer les types de champs
  types <- unlist(lapply(d, function(x) {
    class(x)[1]
  }))

  # le type liste n'est pas géré :
  attempt::stop_if(any(grepl("list", types)), isTRUE, msg = "Le type 'list' n'est pas g'\u00e9r\u00e9.\nveuillez convertir les champs concern\u00e9 en texte.")

  ## modular handling of different data type attributes (add new below)
  ### gerer les attributs temporels
  attr2 <- lapply(d[1, ], function(x) {
    attr(x, "tzone")[1]
  })
  badtz <- unlist(lapply(attr2, function(x) {
    any(is.null(x), !x %in% OlsonNames())
  }))
  attr2[badtz] <- "NULL"
  attr2 <- unlist(attr2)

  ### convert non-matching tz time to db tz (runs but values unchanged in posixlt objects)
  pgtz <- DBI::dbGetQuery(con, "SHOW timezone;")[1, 1]
  tzl <- names(attr2[attr2 != "NULL" & attr2 != pgtz])
  for (t in tzl) {
    eval(parse(text = paste0("attributes(d$", t, ")$tzone <- pgtz")))
  }

  ### Gerer les facteurs
  fact <- unlist(lapply(d[1, ], function(x) {
    paste0("/*/", paste(attr(x, "levels"), collapse = "/*/"),
           "/*/")
  }))
  fact <- gsub(",", "\\,", fact, fixed = TRUE)
  attr2[!fact == "/*//*/"] <- fact[!fact == "/*//*/"]
  ### fin factor

  ## Fabriquer la table des types et attributs
  defs <- paste0("{{", paste(names(d), collapse = ","), "},{",
                 paste(as.character(types), collapse = ","), "},{",
                 paste(as.character(attr2), collapse = ","), "},{",
                 paste(noms_r, collapse = ","), "}}")

  defs2 <- data.frame(table_nm = table, df_def = defs)
  suppressWarnings(suppressMessages({
    rpostgis::pgInsert(con, c(schema, "zz_r_df_def"), defs2, upsert.using = "table_nm",
                       row.names = FALSE)
  }))

  # precaution d'encodage des variables de type texte ou categorielles---------------
  d <- dplyr::mutate(d, dplyr::across(tidyselect::vars_select_helpers$where(is.character), enc2utf8),
                     dplyr::across(tidyselect::vars_select_helpers$where(is.factor), ~ forcats::fct_relabel(.x, enc2utf8)))

  # nom des variables de la cle primaire---------
  if(is.null(pk)) {
    d <- dplyr::mutate(d, id_row__ = 1:nrow(d)) %>%
      dplyr::relocate(.data$id_row__)
    nom_var_cle = "id_row__"
  } else if(length(pk) == 1 & pk[1] == 1) {
    nom_var_cle = names(d)[1]
  } else {
    d <- dplyr::relocate(d, tolower(pk))
    nom_var_cle = tolower(pk)
  }

  ## Ecriture de la table : un comportement different pour les donnees spatiales et les autres dataframes------
  if( any(grepl("sf", class(d))) ) {
    nom_col_geo <- attributes(d)$sf_column
    attempt::warn_if(nom_col_geo != names(d)[ncol(d)],
                     msg = "La colonne geo n'est pas en fin de dataset, c'est une mauvaise pratique.")
    sf::st_write(obj = d, dsn = con, delete_layer = overwrite, layer = c(schema = schema, table = table))
    message(paste0('Table ', table,' ecrite sur le serveur avec sf::st_write(). '))
    # Ajout contrainte forcage crs ouverture couche
    crs <- sf::st_crs(x = d, parameters = TRUE)$epsg
    crs <- ifelse(is.null(crs), 2154, crs)
    sql_query_crs <- paste0("ALTER TABLE ", schema, ".", table ," ADD CONSTRAINT enforce_srid_geom CHECK (st_srid(", nom_col_geo, ") = ", crs, ");")
    DBI::dbExecute(conn = con, sql_query_crs)
    message(paste0("Ajout de la contrainte du CRS d'ouverture : ", crs, "."))
    # Ajout contrainte nombre de dimension
    # if(chek_dim) {
    dim <- max(unique(sf::st_dimension(d, NA_if_empty = TRUE)), 2, na.rm = TRUE)
    sql_query_dim <- paste0("ALTER TABLE ", schema, ".", table ," ADD CONSTRAINT enforce_dims_geom CHECK (st_ndims(", nom_col_geo, ") = ", dim, ");")
    DBI::dbExecute(conn = con, sql_query_dim)
    message(paste0("Ajout de la contrainte du nombre de dimension : ", dim, "."))

    # }

    # Ajout index geo
    index_spatial <- rpostgis::dbIndex(conn = con, name = c(schema, table), colname = nom_col_geo,
                                       idxname = paste0("gix_", table), unique = FALSE, method = "gist",
                                       display = FALSE, exec = TRUE)
    mess <- "spatial "
    mess2 <- ifelse(index_spatial, paste0(" - index spatial declar\u00e9 sur '", nom_col_geo, "'"), "")
  } else {
    DBI::dbWriteTable(con, name = c(schema, table), d, row.names = FALSE, overwrite = overwrite)
    mess <- ""
    mess2 <- ""
  }
  suppressMessages({
    rpostgis::dbAddKey(con, c(schema, table), colname = nom_var_cle, type = "primary")
  })

  # ajout de metadonnees basiques en commentaires de la table-----------
  meta <- enc2utf8(paste0("Table post\u00e9e le ", format(Sys.Date(), '%d/%m/%Y'), ", par ", Sys.getenv("USERNAME"), "."))
  suppressMessages({
    rpostgis::dbComment(conn = con, name = c(schema, table), comment = meta, type = "table", exec = TRUE)
  })

  # ajout des droits par defaut du schema a la table postee



  DBI::dbDisconnect(con)
  rm(con)

  # information de l'utilisateur-----------
  message("Dataframe ", mess, "\u00e9crit dans la table ", paste0(c(schema, table), collapse = "."), " : ")
  message(" - commentaire ajout\u00e9 : '", meta, "'")
  message(" - cl\u00e9 primaire declar\u00e9e sur : '", paste0(nom_var_cle, collapse = "', '"), "'")
  message(mess2)

  return(invisible(NULL))

}

#
# poster_data(data = ecln_cogifie, table = "test_ecln_cog", schema = "public", db = "public", server = "localhost",
#             pk = c("TypeZone", "CodeZone", "date"), post_row_name = FALSE, overwrite = TRUE, user = "does")
# poster_data(data = iris, table = "test_iris", schema = "public", db = "public", server = "localhost",
#             pk = NULL, post_row_name = TRUE, overwrite = TRUE, user = "does")
# poster_data(data = COGiter::departements_geo, table = "test_dep", schema = "public", db = "public", server = "localhost",
#             pk = "DEP", post_row_name = FALSE, overwrite = TRUE, user = "does")
