library(gmailr)
library(logging)
library(base64enc)
library(dplyr)

# Configuration du logging
basicConfig(level = 'DEBUG')

# Fonction pour encoder le sujet en Base64
encode_base64_utf8 <- function(string) {
  raw_string <- charToRaw(string)
  encoded_string <- base64encode(raw_string)
  return(sprintf("=?UTF-8?B?%s?=", encoded_string))
}

html_body <- paste(readLines("data-raw/datamartage.html"), collapse = "\n")

# Fonction pour envoyer un email  #,
send_email <- function(subject, body, attachment_path, to = "adl.dreal-pdl@developpement-durable.gouv.fr", cc = "lisa.smah@developpement-durable.gouv.fr") {
  gm_auth_configure(path = "X:/SCTE/CSD/APPLICATIONS_SIG_BUREAUTIQUES/RESSOURCES_R/client_secret_api_gmail_csd.json")
  gm_auth(
    scopes = c(
      "https://www.googleapis.com/auth/gmail.send",
      "https://www.googleapis.com/auth/gmail.compose"
    )
  )

  # Encoder le sujet en Base64 pour éviter les problèmes d'encodage
  subject_base64 <- encode_base64_utf8(subject)

  email <- gm_mime() %>%
    gm_to(to) %>%
    gm_cc(cc) %>%
    gm_from("csd.scte.dreal.pdl@gmail.com") %>%
    gm_subject(subject_base64) %>%
    gm_text_body(body) %>%
    gm_html_body(body) %>%
    gm_attach_file(attachment_path)

  gm_send_message(email)
}

# Fonction pour lire et filtrer les logs
read_logs <- function(log_file) {
  logs <- readLines(log_file)
  #Filtrer les lignes indésirables
  filtered_logs <- logs[!grepl("Attaching core tidyverse packages", logs) &
                          !grepl("Conflicts", logs) &
                          !grepl("package ‘", logs) &
                          !grepl("✔", logs) &
                          !grepl("ℹ Use the conflicted package", logs) &
                          !grepl("─", logs) &
                          !grepl("✖", logs)&
                          !grepl("Messages d'avis", logs) &
                          !grepl("compilé avec la version R", logs) &
                          !grepl("Attachement du package", logs) &
                          !grepl("L'objet suivant est masqué depuis", logs)&
                          !grepl("complete", logs)]
  return(filtered_logs)
}

# Fonction pour analyser les logs et préparer le contenu de l'email
prepare_email_content <- function(logs) {
  tryCatch({
    datamart_success <- any(grepl("Mise à jour du datamart catalogueR réussie", logs))
    deploy_success <- any(grepl("Déploiement de l'application réussi", logs))
    error_messages <- logs[grepl("Erreur|Échec", logs)]

    taille_datamart <- logs[grepl("Taille du datamart", logs)]
    temps_execution <- logs[grepl("Temps d'exécution", logs)]

    date <- Sys.Date()
    if (datamart_success && deploy_success) {
      subject <- "Succès mise à jour datamart catalogueR et déploiement de l'application"
      body <- paste(logs, collapse = "\n")
    } else if (!datamart_success) {
      subject <- "Échec mise à jour datamart catalogueR"
      body <- sprintf("La mise à jour du datamart a échoué le %s. Erreurs:\n%s", date, paste(error_messages, collapse = "\n"))
    } else if (!deploy_success) {
      subject <- "Échec du déploiement de l'application"
      body <- sprintf("Le déploiement de l'application a échoué le %s. Erreurs:\n%s", date, paste(error_messages, collapse = "\n"))
    }

    list(subject = subject, body = body)
  }, error = function(e) {
    subject <- "Erreur lors de l'analyse des logs"
    body <- sprintf("Une erreur est survenue lors de l'analyse des logs le %s. Erreur:\n%s", Sys.Date(), e$message)
    list(subject = subject, body = body)
  })
}

# Chemin vers le fichier de log généré par script_routine
args <- commandArgs(trailingOnly = FALSE)
script_path <- sub("--file=", "", args[grep("--file=", args)])
project_path <- normalizePath(file.path(dirname(script_path), ".."))
setwd(project_path)
log_file <- "logs/script_routine.log"

# Lire et analyser les logs
logs <- read_logs(log_file)

# Préparer le sujet et le contenu de l'email basé sur l'analyse des logs
email_content <- prepare_email_content(logs)
html_file_path <- "data-raw/datamartage.html"
# Envoi de l'email de notification
send_email(email_content$subject, email_content$body,html_file_path)

