
#' table_prod : table des indicateurs d'evolution des productions renouvelables annuelles par filiere aux echelles regionale et France entiere
#'
#' @param df le df pret pour la visualisation ggplot, produit par `prep_prod()`
#' @param an l'annee observee au format numerique
#' @param obj_reg liste indiquant les objectifs de production en TWh fixes pour la region a l'horizon 2023, par exemple `list(eol = 3.8, pv = 0.58, hydro = 0.025, elec = NA_real_)`. 'eol' designe la filiere eolienne, 'pv' le solaire photovoltaïque, 'hydro' l'hydraulique et 'elec' la production cible de l'ensemble des filieres electriques renouvelables
#' @param obj_nat les objectifs de production en TWh fixes a l'echelle nationale a l'horizon 2023, par exemple `list(eol = 39.9, pv = 5.9, hydro = 64.4, elec = 148)`. 'eol' designe la filiere eolienne, 'pv' le solaire photovoltaïque, 'hydro' l'hydraulique et 'elec' la production cible de l'ensemble des filieres electriques renouvelables
#' 
#'
#' @return Un dataframe des indicateurs principaux a prendre en compte pour le commentaire sur les productions renouvelables
#' @export

#' @importFrom rlang .data
#' @importFrom dplyr filter arrange mutate select group_by lead ungroup summarise first last bind_rows lag case_when across rename_with
#' @importFrom tidyselect vars_select_helpers
#'
#' @examples
#' \dontrun{
#' table_prod(df = prod_enr_viz, an = 2020)
#' }
table_prod <- function(df, an = 2020, obj_reg = list(eol = NA_real_, pv = NA_real_, hydro = NA_real_, elec = NA_real_), 
                       obj_nat = list(eol = 52, pv = 24, hydro = 62, elec = 155)) {
  
  dix_ans_avt <- an - 9
  
  tble_prod_elec <- df %>% 
    dplyr::filter(!grepl("gaz", .data$indicateur), .data$annee >= dix_ans_avt) %>%
    dplyr::arrange(.data$annee, .data$region, .data$filiere) %>% 
    dplyr::mutate(entete = paste0(.data$indicateur, " - ", .data$region), annee = as.character(.data$annee)) %>% 
    dplyr::select("entete", "indicateur", "annee", "prod_TWh") %>% 
    dplyr::group_by(.data$annee, .data$indicateur) %>% 
    dplyr::arrange(.data$entete) %>% 
    dplyr::mutate(part_reg = .data$prod_TWh/dplyr::lead(.data$prod_TWh)*100) %>% 
    dplyr::ungroup() %>%
    dplyr::group_by(.data$entete, .data$indicateur) 
  
  tble_prod_elec2 <- dplyr::bind_rows(tble_prod_elec, dplyr::summarise(tble_prod_elec, 
                                                                       annee = paste0("Moy ", dplyr::first(.data$annee), " - ", dplyr::last(.data$annee)),
                                                                       prod_TWh = mean(.data$prod_TWh), .groups = "drop")) %>%   
    dplyr::group_by(.data$entete, .data$indicateur) %>% 
    dplyr::filter(.data$annee >= as.character(an - 1) | .data$annee == as.character(dix_ans_avt)) %>% 
    dplyr::mutate(evol_n_nmoins1 = ((.data$prod_TWh - dplyr::lag(.data$prod_TWh))/dplyr::lag(.data$prod_TWh)*100),
                  evol_partreg_n_nmoins1 = ((.data$part_reg - dplyr::lag(.data$part_reg))/dplyr::lag(.data$part_reg)*100),
                  evol_n_nmoins9 = ((.data$prod_TWh - dplyr::lag(.data$prod_TWh, 2))/dplyr::lag(.data$prod_TWh, 2)*100),
                  ecart_moy = ((.data$prod_TWh - dplyr::lead(.data$prod_TWh))/dplyr::lead(.data$prod_TWh)*100 ),
                  obj_2023 = dplyr::case_when(
                    .data$entete == "production_eolienne_renouvelable - 1reg" ~ obj_reg$eol,
                    .data$entete == "production_eolienne_renouvelable - 2FR" ~ obj_nat$eol,
                    .data$entete == "production_solaire_renouvelable - 1reg" ~ obj_reg$pv,
                    .data$entete == "production_solaire_renouvelable - 2FR" ~ obj_nat$pv,
                    .data$entete == "production_hydraulique_renouvelable - 1reg" ~ obj_reg$hydro,
                    .data$entete == "production_hydraulique_renouvelable - 2FR" ~ obj_nat$hydro,
                    .data$entete == "production_electrique_renouvelable - 2FR" ~ obj_nat$elec,
                    TRUE ~ NA_real_),
                  tx_realis_obj_2023 = .data$prod_TWh/.data$obj_2023*100,
                  dplyr::across(tidyselect::vars_select_helpers$where(is.numeric), ~ round(.x, 1))) %>% 
    dplyr::arrange(.data$indicateur) %>% 
    dplyr::ungroup()
  
  tble_prod_elec3 <- dplyr::filter(tble_prod_elec2, .data$annee == as.character(an)) %>% 
    dplyr::select(-"annee", -"indicateur") %>% 
    dplyr::rename_with( ~ gsub("_n_", paste0(" ", an, " "), .x) %>% 
                   gsub("nmoins1", an-1, .) %>% 
                   gsub("nmoins9", dix_ans_avt, .) %>% 
                   gsub("moy", paste0("moyenne ", dix_ans_avt, "-", an), .) %>% 
                   gsub("_", " ", .)) 

  return(tble_prod_elec3)
  
}


#' Visualisation de la table des indicateurs d'evolution des productions renouvelables annuelles par filiere aux echelles regionale et France entiere
#'
#' @param df  le df pret pour la visualisation flextable, produit par `table_prod()`
#'
#' @return  le widget flextable des indicateurs d'evolution des productions renouvelables
#' @importFrom dplyr mutate across
#' @importFrom flextable flextable set_header_labels set_table_properties bg color
#' @importFrom tidyr starts_with
#' @export
#'
#' @examples
#' \dontrun{
#' table_prod_viz(df = tble_prod_elec2)
#' }

table_prod_viz <- function(df) {
  
  df %>% 
    dplyr::mutate(entete = gsub("production_|_renouvelable|1|2", "", .data$entete),
                  dplyr::across(tidyr::starts_with(c("part", "evol", "ecart", "tx")), ~ ifelse(is.na(.x), .x, paste(.x, "%")))) %>% 
    flextable::flextable() %>%
    flextable::set_header_labels(entete = "indicateur - territoire") %>%
    flextable::set_table_properties(layout = "autofit") %>%
    flextable::bg(i = 1, bg = "#7D4E5B", part = "header") %>%
    flextable::color(i = 1, color = 'white', part = "header")
}
