
#' table_biogaz : table des indicateurs d'evolution des production de biomethane aux echelles regionale et France entiere
#'
#' @param df le df pret pour la visualisation ggplot, produit par `prep_prod()`
#' @param an l'annee observee au format numerique
#'
#' @return Un dataframe des indicateurs principaux a prendre en compte pour le commentaire sur les productions de biométhane
#' @export

#' @importFrom dplyr filter pull arrange mutate select group_by lead ungroup lag across rename_with
#' @importFrom rlang .data
#' @importFrom tidyselect vars_select_helpers
#'
#' @examples
#' \dontrun{
#' table_biogaz(df = prod_enr_viz, an = 2020)
#' }
table_biogaz <- function(df, an = 2020) {
  
  annee_debut_biogaz_reg <- dplyr::filter(df, grepl("gaz", .data$indicateur), .data$region == "1reg", .data$prod_TWh != 0) %>% 
    dplyr::pull(.data$annee) %>% min()
  
  tble_prod_biogaz <- df %>% 
    dplyr::filter(grepl("gaz", .data$indicateur), .data$annee >= annee_debut_biogaz_reg) %>%
    dplyr::arrange(.data$annee, .data$region, .data$filiere) %>% 
    dplyr::mutate(entete = paste0(.data$indicateur, " - ", .data$region), annee = as.character(.data$annee)) %>% 
    dplyr::select("entete", "zone", "indicateur", "annee", "prod_TWh") %>% 
    dplyr::group_by(.data$annee, .data$indicateur) %>% 
    dplyr::arrange(.data$entete) %>% 
    dplyr::mutate(part_reg = .data$prod_TWh/dplyr::lead(.data$prod_TWh)*100) %>% 
    dplyr::ungroup() %>%
    dplyr::group_by(.data$entete, .data$indicateur) %>% 
    dplyr::filter(.data$annee >= as.character(an - 1) | .data$annee == as.character(annee_debut_biogaz_reg)) %>% 
    dplyr::mutate(evol_n_nmoins1 = ((.data$prod_TWh - dplyr::lag(.data$prod_TWh))/dplyr::lag(.data$prod_TWh)*100),
           evol_partreg_n_nmoins1 = ((.data$part_reg - dplyr::lag(.data$part_reg))/dplyr::lag(.data$part_reg)*100),
           evol_n_debut_biogaz_reg = ((.data$prod_TWh - dplyr::lag(.data$prod_TWh, 2))/dplyr::lag(.data$prod_TWh, 2)*100),
           dplyr::across(tidyselect::vars_select_helpers$where(is.numeric), ~ round(.x, 1))) %>% 
    dplyr::ungroup() %>% 
    dplyr::filter(.data$annee == as.character(an)) %>% 
    dplyr::arrange(.data$entete) %>% 
    dplyr::select(-"annee", -"entete") %>% 
    dplyr::rename_with( ~ gsub("_n_", paste0(" ", an, " "), .x) %>% 
                   gsub("nmoins1", an-1, .) %>% 
                   gsub("debut_biogaz_reg", annee_debut_biogaz_reg, .) %>% 
                   gsub("_", " ", .))
  
  
  return(tble_prod_biogaz)
  
}


#' Visualisation de la table des indicateurs d'evolution des production de biomethane aux echelles regionale et France entiere
#'
#' @param df  le df pret pour la visualisation flextable, produit par `table_biogaz()`
#'
#' @return  le widget flextable des indicateurs d'evolution des production de biomethane
#' @importFrom dplyr mutate across
#' @importFrom flextable flextable set_header_labels set_table_properties bg color
#' @importFrom tidyr starts_with
#' @export
#'
#' @examples
#' \dontrun{
#' table_biogaz_viz(df = table_biogaz)
#' } 

table_biogaz_viz <- function(df) {
  
    dplyr::mutate(df, dplyr::across(tidyr::starts_with(c("part", "evol")), ~ ifelse(is.na(.x), .x, paste(.x, "%")))) %>% 
    flextable::flextable() %>%
    flextable::set_header_labels(zone = "injection de biom\u00e9thane") %>%
    flextable::set_table_properties(layout = "autofit", ) %>%
    flextable::bg(i = 1, bg = "#7D4E5B", part = "header") %>%
    flextable::color(i = 1, color = 'white', part = "header")
  
}
