#' prep_dju : preparation de l'indice de rigueur pour integration dans le graphe des consommations
#'
#' @param url_dju_nat : chaine de caractere, fin de l'url qui pointe vers le fichier excel des DJU France metropolitaine publie par le SDES (ce qu'il y a après "https://www.statistiques.developpement-durable.gouv.fr/media/")
#' @param url_dju_reg : chaine de caractere, fin de l'url qui pointe vers le fichier excel des DJU France regionaux publie par le SDES, pour avoir tous l'historique utiliser le fichier des vieilles regions (ce qu'il y a après "https://www.statistiques.developpement-durable.gouv.fr/media/")
#' @param reg : le code region au format texte
#' @param an : l'annee observee au format numerique
#' @param coeff_dju : un coefficient qui permet de ramener l'indice de rigueur (oscillant autour de 1) dans l'echelle des consommations regionales (oscillant entre 2 et 70 TWh selon les region), 25 par exemple pour les Pays de la Loire
#'
#' @return un dataframe presentant l'indice de rigueur pour la France et la region observee, depuis 2013, pret a être incorpore dans le graphes des consommations
#' 
#' @importFrom attempt stop_if
#' @importFrom curl curl_download
#' @importFrom dplyr mutate select arrange group_by bind_rows summarise filter first ungroup rename_with if_else across
#' @importFrom readxl read_xlsx
#' @importFrom tidyr separate pivot_longer pivot_wider unnest
#' @importFrom tidyselect vars_select_helpers
#' @importFrom tricky str_standardize set_standard_names
#' @importFrom rlang .data
#' @importFrom stringr str_extract
#' @importFrom utils download.file
#' @export
#'
#' @examples
#' prep_dju(reg = "52", an = 2024, coeff_dju = 25)

prep_dju <- function(url_dju_nat = NULL, url_dju_reg = NULL, reg = "52", 
                     an = 2024, coeff_dju = 25) {
  path <- tempdir()
  url_base <- "https://www.statistiques.developpement-durable.gouv.fr/media/"
  if(is.null(url_dju_nat)) {
    url_dju_nat <- "205/download?inline"
    }
  if(is.null(url_dju_reg)) {
    url_dju_reg <- "206/download?inline"
  }
  url_dju_nat_full <- paste0(url_base, url_dju_nat)
  url_dju_reg_full <- paste0(url_base, url_dju_reg)
  
  curl::curl_download(url = url_dju_nat_full, destfile = paste0(path, "/dju_nat.xlsx"))
  curl::curl_download(url = url_dju_reg_full, destfile = paste0(path, "/dju_reg.xlsx"))
  
  period_ref <- readxl::read_xlsx(path =  paste0(path, "/dju_nat.xlsx"), sheet = "DJU 17\u00b0C", range = "A5:A5",  col_names = "periode_normale") %>% 
    dplyr::mutate(bornes = stringr::str_extract(string = .data$periode_normale, pattern = "[[:digit:]]{4}-[[:digit:]]{4}")) %>% 
    tidyr::separate(col = "bornes", into = c("min", "max"), sep = "-", remove = FALSE)
  
  attempt::stop_if(all(grepl("[[:digit:]]{4}", period_ref$min), grepl("[[:digit:]]{4}", period_ref$max)), isFALSE, 
                   msg = "Les bornes de la periode de reference n\'ont pas pu \u00eatre d\u00e9tect\u00e9es dans le fichier DJU du SDES, signalez ce bug \u00e0 l\'administrateur du package.")

  dju_nat <- readxl::read_xlsx(path =  paste0(path, "/dju_nat.xlsx"), sheet = "DJU 17\u00b0C", skip = 3, n_max = 3) %>% 
    dplyr::mutate(var = tricky::str_standardize(.data$...1) %>%
                    gsub("dju_de_reference_moyenne_sur_la_periode_de_reference_", paste0("dju_ref_", period_ref$bornes), .),
                  nom_region = "France m\u00e9tropolitaine", code_region = "00") %>% 
    tidyr::pivot_longer(cols = -c("nom_region", "code_region", "var", "...1"), names_to = "annee", values_to = "valeur",
                        names_transform = list(annee = as.integer) ) %>% 
    dplyr::select(-"...1") %>% 
    tidyr::pivot_wider(names_from = "var", values_from = "valeur")
  
  dju_reg0 <- readxl::read_xlsx(path =  paste0(path, "/dju_reg.xlsx"), sheet = "DJU 17\u00b0C", skip = 3, n_max = 22 ) %>%
    tidyr::pivot_longer(cols = -c(1:2), names_to = "annee", values_to = "valeur", names_transform = list(annee = as.integer)) %>%
    tricky::set_standard_names() %>%
    dplyr::mutate(var = "dju", code_region = as.character(.data$code_region)) %>%
    dplyr::arrange(.data$code_region, .data$annee) %>% 
    # on renomme la colonne nom_ancienne_region en nom region si elle est presente 
    dplyr::rename_with(~gsub("nom_ancienne_region", "nom_region", .x))
  
  dju_reg_ref <- dplyr::filter(dju_reg0, .data$annee %in% c(period_ref$min:period_ref$max)) %>% 
    dplyr::group_by(.data$code_region, .data$nom_region) %>%
    dplyr::summarise(valeur = mean(.data$valeur), var = "dju_moy_ref", annee = dplyr::first(.data$annee), 
                     .groups = "drop_last") %>%
    dplyr::mutate(annee = list(.data$annee:an)) %>% 
    tidyr::unnest(cols = "annee")
                                  
  dju_reg <- dplyr::bind_rows(dju_reg0, dju_reg_ref) %>%
    tidyr::pivot_wider(names_from = "var", values_from = "valeur") %>%
    dplyr::mutate(indice_de_rigueur = .data$dju/.data$dju_moy_ref) %>%
    dplyr::rename_with(~gsub("dju_moy_ref", paste0("dju_ref_", period_ref$bornes), .x)) %>% 
    dplyr::filter(.data$code_region == reg)
  
  resultat <- dplyr::bind_rows(dju_nat, dju_reg) %>% 
    dplyr::filter(.data$annee>= 2013) %>%
    dplyr::select("annee", region2 = "nom_region", conso_TWh = "indice_de_rigueur") %>%
    dplyr::mutate(annee = as.numeric(.data$annee),
                  nrj = "indice de rigueur",
                  conso_comp = .data$conso_TWh * coeff_dju,
                  region3 = dplyr::if_else(.data$region2 == "France m\u00e9tropolitaine", "2FR", "1reg")                  ,
                  dplyr::across(tidyselect::vars_select_helpers$where(is.numeric), ~ round(.x, 2)))
  
  file.remove(paste0(path, "/dju_nat.xlsx"), paste0(path, "/dju_reg.xlsx"))
  
  return(resultat)
}
